#ifndef AHLGREN_TRIE
#define AHLGREN_TRIE

#include <iostream>
#include <string>
#include <map>
#include <vector>

using namespace std;


template <typename Ctype>
class Trie {
public:
	typedef std::map<char,Trie*> children_t; // each Trie node has a map define char -> children mapping
	typedef typename children_t::size_type size_type;
	typedef vector<Ctype> vec_type;
	typedef typename vec_type::iterator value_iterator;
	typedef typename vec_type::const_iterator value_const_iterator;
	// Consructors / Destructor
	explicit Trie(char s = '\0') : c(s), visitc(0), vptr() { }
	~Trie();
	// Get symbol of node
	char symbol() const { return c; }
	// Does node represent a valid string (or only a prefix)?
	bool is_entry() const { return visitc != 0; }
	bool is_visited() const { return visitc != 0; }
	// Number of times the node has been visited
	unsigned visits() const { return visitc; }
	// Normalize visits: keep 0 if 0, set to 1 if greater than zero
	void normalize_visits();
	// Get number of valid entries
	size_type size() const;
	// Get number of nodes
	size_type nodes() const;
	// Get total number of visits
	size_type total_visits() const;
	// Return pointer to node if present, nullptr otherwise
	Trie* contains(const string& expr) { return const_cast<Trie*>(const_cast<const Trie*>(this)->contains(expr)); }
	const Trie* contains(const string& expr) const { return contains(expr,expr.begin()); }
	// Get Ctype range of iterators for specific symbol
	pair<value_const_iterator,value_const_iterator> equal_range(const string& s) const {
		const Trie* node = contains(s);
		if (!node) return make_pair(vptr.end(),vptr.end());
		return node->equal_range();
	}
	// Get Ctype range for this trie node
	pair<value_iterator,value_iterator> equal_range() { return make_pair(value_begin(),value_end()); }
	// Access trie node
	Trie* operator[](const string& expr) { return contains(expr); }
	const Trie* operator[](const string& expr) const { return contains(expr); }
	//const Ctype* operator[](const string& expr) const;
	// Access entry node
	Trie* get_entry(const string& s) { Trie* t = this->contains(s); return t && t->is_entry() ? t : nullptr; }
	const Trie* get_entry(const string& s) const { return static_cast<Trie*>(this)->get_entry(s); }

	// Insert expression into trie, return pointer to node
	Trie* insert(const string& expr, const Ctype& p) { return insert(expr,p,expr.begin()); }
	// Print trie tree (can be expensive)
	void print(ostream& os) const { print(os,""); }

	// Node specific operations

	// Navigate Trie by moving down to next node with character ch
	const Trie* next(char ch) const;
	// Set/Get Numerics
	value_iterator value_begin() { return vptr.begin(); }
	value_iterator value_end() { return vptr.end(); }

	value_const_iterator value_begin() const { return vptr.begin(); }
	value_const_iterator value_end() const { return vptr.end(); }
	pair<value_const_iterator,value_const_iterator> equal_range() const { return make_pair(value_begin(),value_end()); }
	
	// Direct manipulation of this node's vector
	void insert_here(const Ctype& v) { vptr.push_back(v); visitc = 1; }
	void insert_here(Ctype&& v) { vptr.push_back(std::move(v)); visitc = 1; }

	// (Low Level) Register one more visit
	void register_visit() { ++visitc; }
protected:
	children_t childv;
	char c;
	unsigned visitc;
	vec_type vptr;
	//bool is_closed;
	void print(ostream&, string expr) const;
	Trie* insert(const string& expr, const Ctype& p, string::const_iterator si);
	Trie* contains(const string& expr, string::const_iterator si) { return const_cast<Trie*>(const_cast<const Trie*>(this)->contains(expr,si)); }
	const Trie* contains(const string& expr, string::const_iterator si) const;
private:
	Trie(const Trie&); // forbid copying
	Trie& operator=(const Trie&); // forbid assignment
};

template <typename Ctype>
inline ostream& operator<<(ostream& os, const Trie<Ctype>& n)
{
	n.print(os);
	return os;
}

template <typename Ctype>
inline const Trie<Ctype>* Trie<Ctype>::next(char ch) const
{
	auto i = childv.find(ch);
	return (i == childv.end()) ? nullptr : i->second;
	//for (auto i = childv.begin(); i != childv.end(); ++i)
	//	if ((*i)->c == ch) return *i;
	//return nullptr; // no child of this for character ch
}


template <typename Ctype>
Trie<Ctype>::~Trie()
{
	for_each(childv.begin(),childv.end(),[](pair<const char,Trie*>& v){delete v.second;});
	//for_each(vptr.begin(),vptr.end(),[](Ctype* p){delete p;});
}

template <typename Ctype>
Trie<Ctype>* Trie<Ctype>::insert(const string& expr, const Ctype& v, string::const_iterator si)
{
	if (si == expr.end()) {
		// no more characters: add here
		vptr.push_back(v);
		visitc = 1;
		return this;
	}

	auto i = childv.find(*si);
	if (i != childv.end()) {
		// Node already exists for this character
		return i->second->insert(expr,v,++si);
	} else {
		// Not found: create new node
		auto at = childv.insert( make_pair(*si,new Trie(*si)) );
		return (*at.first).second->insert(expr,v,++si); // recursive call
	}
}

template <typename Ctype>
const Trie<Ctype>* Trie<Ctype>::contains(const string& expr, string::const_iterator si) const
{
	if (si == expr.end()) {
		// no more characters
		return this;
	}

	auto i = childv.find(*si);
	if (i != childv.end()) {
		// Node already exists
		return i->second->contains(expr,++si);
	} else {
		// Not found
		return nullptr;
	}
}

template <typename Ctype>
void Trie<Ctype>::normalize_visits()
{
	if (visitc > 1) visitc = 1;
	for_each(childv.begin(),childv.end(),[](pair<const char,Trie*>& v){v.second->normalize_visits();});
}

template <typename Ctype>
typename Trie<Ctype>::size_type Trie<Ctype>::size() const
{
	size_type n = is_visited();
	for_each(childv.begin(),childv.end(),[&](Trie* t){n += t->size();});
	return n;
}

template <typename Ctype>
typename Trie<Ctype>::size_type Trie<Ctype>::total_visits() const
{
	size_type n = visitc;
	for_each(childv.begin(),childv.end(),[&](Trie* t){n += t->total_visits();});
	return n;
}

template <typename Ctype>
typename Trie<Ctype>::size_type Trie<Ctype>::nodes() const
{
	size_type n = 1;
	for_each(childv.begin(),childv.end(),[&](Trie* t){n += t->nodes();});
	return n;
}

template <typename Ctype>
void Trie<Ctype>::print(ostream& os, string expr) const
{
	if (c != '\0') expr += c;
	if (is_visited()) {
		os << expr << " [visits: " << visitc << "]\n";
	}
	for_each(childv.begin(),childv.end(),[&](Trie* t){t->print(os,expr);});
}



#endif
